--[[
# Element: Group Role Indicator

Toggles the visibility of an indicator based on the unit's current group role (tank, healer or damager).

## Widget

GroupRoleIndicator - A `Texture` used to display the group role icon.

## Notes

A default texture will be applied if the widget is a Texture and doesn't have a texture or a color set.

## Examples

	-- Position and size
	local GroupRoleIndicator = self:CreateTexture(nil, 'OVERLAY')
	GroupRoleIndicator:SetSize(16, 16)
	GroupRoleIndicator:SetPoint('LEFT', self)

	-- Register it with oUF
	self.GroupRoleIndicator = GroupRoleIndicator
--]]
local parent, ns = ...
local oUF = ns.oUF

local UnitGroupRolesAssigned = UnitGroupRolesAssigned

local roleCoords = {
	TANK = {0, 0.30, 0.34, 0.64},
	HEALER = {0.31, 0.61, 0.02, 0.31},
	DAMAGER = {0.31, 0.61, 0.34, 0.64}
}

local function Update(self, event)
	local element = self.GroupRoleIndicator

	--[[ Callback: GroupRoleIndicator:PreUpdate()
	Called before the element has been updated.

	* self - the GroupRoleIndicator element
	--]]
	if (element.PreUpdate) then
		element:PreUpdate()
	end

	--"Interface\\LFGFrame\\LFGRole.blp"
	local isTank, isHealer, isDamage = UnitGroupRolesAssigned(self.unit)
	if (isTank or isHealer or isDamage) then
		local role = isTank and "TANK" or isHealer and "HEALER" or isDamage and "DAMAGER"
		element:SetTexCoord(roleCoords[role])
		element:Show()
	else
		element:SetTexture(nil)
		element:Hide()
	end

	--[[ Callback: GroupRoleIndicator:PostUpdate(role)
	Called after the element has been updated.

	* self - the GroupRoleIndicator element
	* role - the role as returned by [UnitGroupRolesAssigned](http://wowprogramming.com/docs/api/UnitGroupRolesAssigned.html)
	--]]
	if (element.PostUpdate) then
		return element:PostUpdate(role)
	end
end

local function Path(self, ...)
	--[[ Override: GroupRoleIndicator.Override(self, event, ...)
	Used to completely override the internal update function.

	* self  - the parent object
	* event - the event triggering the update (string)
	* ...   - the arguments accompanying the event
	--]]
	return (self.GroupRoleIndicator.Override or Update)(self, ...)
end

local function ForceUpdate(element)
	return Path(element.__owner, "ForceUpdate")
end

local function Enable(self)
	local element = self.GroupRoleIndicator
	if (element) then
		element.__owner = self
		element.ForceUpdate = ForceUpdate

		if (self.unit == "player") then
			self:RegisterEvent("PLAYER_ROLES_ASSIGNED", Path, true)
		else
			self:RegisterEvent("PARTY_MEMBERS_CHANGED", Path, true)
		end

		if (element:IsObjectType("Texture") and not element:GetTexture()) then
			element:SetTexture([[Interface\LFGFrame\UI-LFG-ICON-PORTRAITROLES]])
		end

		return true
	end
end

local function Disable(self)
	local element = self.GroupRoleIndicator
	if (element) then
		element:Hide()

		self:UnregisterEvent("PLAYER_ROLES_ASSIGNED", Path)
		self:UnregisterEvent("PARTY_MEMBERS_CHANGED", Path, true)
	end
end

oUF:AddElement("GroupRoleIndicator", Path, Enable, Disable)